/*******************************************************************************

 Project Name:      Grove OLED LCD 1.12.cprj
 Required Libs's:   IntFunc_lib.cc
 Files:             Grove OLED LCD 1.12.cc
 Writer:            CCPRO-TEAM
 Date:              10.05.2015
 Function:          Demonstrates the Grove OLED LCD 1.12

 ------------------------------------------------------------------------------
 AVR32 Serie:
 ------------
 Required the C-Control PRO AVR32-Bit UNIT Conrad BN: 192573
 Applicationboard Conrad BN: 192587
 Or Mainboard Conrad BN: 192702

 MEGA Serie:
 -----------
 Required the C-Control PRO MEGA32 UNIT Conrad BN: 198206
 and Evalationboard Conrad BN: 198245
 Or Projectboard Conrad BN: 197287

 Or C-Control PRO MEGA128 UNIT Conrad BN: 198219
 Or C-Control PRO MEGA128CAN Conrad BN: 197989
 and Evaluationboard Conrad BN: 198258
 or Projectboard Conrad BN: 197313

 Connection:
 -----------
 - Connect the sensor red wire to +5V (MEGA) or +3,3V (AVR32)
 - Connect the sensor black wire to GND
 - Connect the sensor yellow wire (I2C SCL) to C-Control PD0 (MEGA128) or PC0(MEGA32)
 - Connect the sensor white wire (I2C SDA) to C-Control PD1 (MEGA128) or PC1(MEGA32)

 Start the program with F10 (or the yellow flash icon) for debug outputs!

 ------------------------------------------------------------------------------

 Note:
 -----
 Source: http://www.seeedstudio.com/depot/Grove-OLED-Display-112-p-781.html

 Grove - OLED Display 1.12" is an OLED monochrome 12864dot matrix display
 module with Grove 4pin I2C Interface .The characteristics of this display
 module are high brightness, self-emission, high contrast ratio, slim outline,
 wide viewing angle, wide temperature range, and low power consumption.
 It has bigger screen and uses all the columns from the SSD1308 chip so that
 it can display more contents than the OLED 9616.

 Features:
 ---------
 * Grove compatible
 * Small size
 * I2C Interface
 * Low power consumption

*******************************************************************************/

#define SeeedOLED_Address_MEGA          0x3C<<1
#define SeeedOLED_Address_AVR32         0x3C

#define SeeedOLED_Max_X                 127 // 128 Pixels
#define SeeedOLED_Max_Y                 63  // 64  Pixels

#define PAGE_MODE                       01
#define HORIZONTAL_MODE                 02

#define SeeedOLED_Command_Mode          0x80
#define SeeedOLED_Data_Mode             0x40
#define SeeedOLED_Display_Off_Cmd       0xAE
#define SeeedOLED_Display_On_Cmd        0xAF
#define SeeedOLED_Normal_Display_Cmd    0xA6
#define SeeedOLED_Inverse_Display_Cmd   0xA7
#define SeeedOLED_Activate_Scroll_Cmd   0x2F
#define SeeedOLED_Dectivate_Scroll_Cmd  0x2E
#define SeeedOLED_Set_Brightness_Cmd    0x81

#define Scroll_Left                     0x00
#define Scroll_Right                    0x01

#define Scroll_2Frames                  0x7
#define Scroll_3Frames                  0x4
#define Scroll_4Frames                  0x5
#define Scroll_5Frames                  0x0
#define Scroll_25Frames                 0x6
#define Scroll_64Frames                 0x1
#define Scroll_128Frames                0x2
#define Scroll_256Frames                0x3

char addressingMode;
byte somedata[10];

// 8x8 Font ASCII 32 - 127 Implemented
// Users can modify this to support more characters(glyphs)
// BasicFont is placed in code memory.

// This font be freely used without any restriction(It is placed in public domain)
flash byte BasicFont[96][8] = {
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x5F,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x07,0x00,0x07,0x00,0x00,0x00,
  0x00,0x14,0x7F,0x14,0x7F,0x14,0x00,0x00,
  0x00,0x24,0x2A,0x7F,0x2A,0x12,0x00,0x00,
  0x00,0x23,0x13,0x08,0x64,0x62,0x00,0x00,
  0x00,0x36,0x49,0x55,0x22,0x50,0x00,0x00,
  0x00,0x00,0x05,0x03,0x00,0x00,0x00,0x00,
  0x00,0x1C,0x22,0x41,0x00,0x00,0x00,0x00,
  0x00,0x41,0x22,0x1C,0x00,0x00,0x00,0x00,
  0x00,0x08,0x2A,0x1C,0x2A,0x08,0x00,0x00,
  0x00,0x08,0x08,0x3E,0x08,0x08,0x00,0x00,
  0x00,0xA0,0x60,0x00,0x00,0x00,0x00,0x00,
  0x00,0x08,0x08,0x08,0x08,0x08,0x00,0x00,
  0x00,0x60,0x60,0x00,0x00,0x00,0x00,0x00,
  0x00,0x20,0x10,0x08,0x04,0x02,0x00,0x00,
  0x00,0x3E,0x51,0x49,0x45,0x3E,0x00,0x00,
  0x00,0x00,0x42,0x7F,0x40,0x00,0x00,0x00,
  0x00,0x62,0x51,0x49,0x49,0x46,0x00,0x00,
  0x00,0x22,0x41,0x49,0x49,0x36,0x00,0x00,
  0x00,0x18,0x14,0x12,0x7F,0x10,0x00,0x00,
  0x00,0x27,0x45,0x45,0x45,0x39,0x00,0x00,
  0x00,0x3C,0x4A,0x49,0x49,0x30,0x00,0x00,
  0x00,0x01,0x71,0x09,0x05,0x03,0x00,0x00,
  0x00,0x36,0x49,0x49,0x49,0x36,0x00,0x00,
  0x00,0x06,0x49,0x49,0x29,0x1E,0x00,0x00,
  0x00,0x00,0x36,0x36,0x00,0x00,0x00,0x00,
  0x00,0x00,0xAC,0x6C,0x00,0x00,0x00,0x00,
  0x00,0x08,0x14,0x22,0x41,0x00,0x00,0x00,
  0x00,0x14,0x14,0x14,0x14,0x14,0x00,0x00,
  0x00,0x41,0x22,0x14,0x08,0x00,0x00,0x00,
  0x00,0x02,0x01,0x51,0x09,0x06,0x00,0x00,
  0x00,0x32,0x49,0x79,0x41,0x3E,0x00,0x00,
  0x00,0x7E,0x09,0x09,0x09,0x7E,0x00,0x00,
  0x00,0x7F,0x49,0x49,0x49,0x36,0x00,0x00,
  0x00,0x3E,0x41,0x41,0x41,0x22,0x00,0x00,
  0x00,0x7F,0x41,0x41,0x22,0x1C,0x00,0x00,
  0x00,0x7F,0x49,0x49,0x49,0x41,0x00,0x00,
  0x00,0x7F,0x09,0x09,0x09,0x01,0x00,0x00,
  0x00,0x3E,0x41,0x41,0x51,0x72,0x00,0x00,
  0x00,0x7F,0x08,0x08,0x08,0x7F,0x00,0x00,
  0x00,0x41,0x7F,0x41,0x00,0x00,0x00,0x00,
  0x00,0x20,0x40,0x41,0x3F,0x01,0x00,0x00,
  0x00,0x7F,0x08,0x14,0x22,0x41,0x00,0x00,
  0x00,0x7F,0x40,0x40,0x40,0x40,0x00,0x00,
  0x00,0x7F,0x02,0x0C,0x02,0x7F,0x00,0x00,
  0x00,0x7F,0x04,0x08,0x10,0x7F,0x00,0x00,
  0x00,0x3E,0x41,0x41,0x41,0x3E,0x00,0x00,
  0x00,0x7F,0x09,0x09,0x09,0x06,0x00,0x00,
  0x00,0x3E,0x41,0x51,0x21,0x5E,0x00,0x00,
  0x00,0x7F,0x09,0x19,0x29,0x46,0x00,0x00,
  0x00,0x26,0x49,0x49,0x49,0x32,0x00,0x00,
  0x00,0x01,0x01,0x7F,0x01,0x01,0x00,0x00,
  0x00,0x3F,0x40,0x40,0x40,0x3F,0x00,0x00,
  0x00,0x1F,0x20,0x40,0x20,0x1F,0x00,0x00,
  0x00,0x3F,0x40,0x38,0x40,0x3F,0x00,0x00,
  0x00,0x63,0x14,0x08,0x14,0x63,0x00,0x00,
  0x00,0x03,0x04,0x78,0x04,0x03,0x00,0x00,
  0x00,0x61,0x51,0x49,0x45,0x43,0x00,0x00,
  0x00,0x7F,0x41,0x41,0x00,0x00,0x00,0x00,
  0x00,0x02,0x04,0x08,0x10,0x20,0x00,0x00,
  0x00,0x41,0x41,0x7F,0x00,0x00,0x00,0x00,
  0x00,0x04,0x02,0x01,0x02,0x04,0x00,0x00,
  0x00,0x80,0x80,0x80,0x80,0x80,0x00,0x00,
  0x00,0x01,0x02,0x04,0x00,0x00,0x00,0x00,
  0x00,0x20,0x54,0x54,0x54,0x78,0x00,0x00,
  0x00,0x7F,0x48,0x44,0x44,0x38,0x00,0x00,
  0x00,0x38,0x44,0x44,0x28,0x00,0x00,0x00,
  0x00,0x38,0x44,0x44,0x48,0x7F,0x00,0x00,
  0x00,0x38,0x54,0x54,0x54,0x18,0x00,0x00,
  0x00,0x08,0x7E,0x09,0x02,0x00,0x00,0x00,
  0x00,0x18,0xA4,0xA4,0xA4,0x7C,0x00,0x00,
  0x00,0x7F,0x08,0x04,0x04,0x78,0x00,0x00,
  0x00,0x00,0x7D,0x00,0x00,0x00,0x00,0x00,
  0x00,0x80,0x84,0x7D,0x00,0x00,0x00,0x00,
  0x00,0x7F,0x10,0x28,0x44,0x00,0x00,0x00,
  0x00,0x41,0x7F,0x40,0x00,0x00,0x00,0x00,
  0x00,0x7C,0x04,0x18,0x04,0x78,0x00,0x00,
  0x00,0x7C,0x08,0x04,0x7C,0x00,0x00,0x00,
  0x00,0x38,0x44,0x44,0x38,0x00,0x00,0x00,
  0x00,0xFC,0x24,0x24,0x18,0x00,0x00,0x00,
  0x00,0x18,0x24,0x24,0xFC,0x00,0x00,0x00,
  0x00,0x00,0x7C,0x08,0x04,0x00,0x00,0x00,
  0x00,0x48,0x54,0x54,0x24,0x00,0x00,0x00,
  0x00,0x04,0x7F,0x44,0x00,0x00,0x00,0x00,
  0x00,0x3C,0x40,0x40,0x7C,0x00,0x00,0x00,
  0x00,0x1C,0x20,0x40,0x20,0x1C,0x00,0x00,
  0x00,0x3C,0x40,0x30,0x40,0x3C,0x00,0x00,
  0x00,0x44,0x28,0x10,0x28,0x44,0x00,0x00,
  0x00,0x1C,0xA0,0xA0,0x7C,0x00,0x00,0x00,
  0x00,0x44,0x64,0x54,0x4C,0x44,0x00,0x00,
  0x00,0x08,0x36,0x41,0x00,0x00,0x00,0x00,
  0x00,0x00,0x7F,0x00,0x00,0x00,0x00,0x00,
  0x00,0x41,0x36,0x08,0x00,0x00,0x00,0x00,
  0x00,0x02,0x01,0x01,0x02,0x01,0x00,0x00,
  0x00,0x02,0x05,0x05,0x02,0x00,0x00,0x00
};

// Bitmap
flash byte SeeedLogo[1024] ={
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x60, 0xf0, 0xc0, 0x00, 0x00, 0x00, 0xfc, 0xff, 0x87, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03,
0xff, 0xfc, 0x00, 0x00, 0x00, 0x80, 0xf0, 0x20, 0x00, 0x00, 0x80, 0xc0, 0xc0, 0x60, 0xe0, 0xc0,
0xc0, 0x00, 0x00, 0x00, 0xc0, 0xc0, 0xc0, 0x60, 0xe0, 0xc0, 0xc0, 0x80, 0x00, 0x00, 0x80, 0xc0,
0xc0, 0xe0, 0x60, 0xc0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0xc0, 0xc0, 0xc0, 0x60, 0xe0, 0xc0, 0xc0,
0x80, 0x00, 0x00, 0x80, 0xc0, 0xc0, 0xe0, 0xe0, 0xc0, 0xc0, 0xf8, 0xf8, 0x00, 0x00, 0x00, 0x00,
0x00, 0xc0, 0xc0, 0xe0, 0x60, 0xc0, 0xc0, 0x80, 0x00, 0xc0, 0xf0, 0xf0, 0xf0, 0xc0, 0x00, 0xc0,
0xc0, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xc0, 0x00, 0x00, 0x80, 0xc0, 0xc0, 0xe0, 0xe0, 0xc0, 0xc0,
0xf8, 0xf8, 0x00, 0xd8, 0xd8, 0x00, 0x00, 0x80, 0xc0, 0xc0, 0xe0, 0x60, 0xc0, 0xc0, 0x80, 0x00,
0x00, 0x03, 0x0f, 0x1e, 0x3c, 0x70, 0xe3, 0xcf, 0x9f, 0x30, 0x00, 0x00, 0x00, 0x00, 0x70, 0xbf,
0xcf, 0xe3, 0x70, 0x78, 0x3e, 0x0f, 0x03, 0x00, 0x00, 0x00, 0x33, 0x77, 0x66, 0x66, 0x66, 0x6c,
0x7d, 0x18, 0x00, 0x1f, 0x3f, 0x76, 0x66, 0x66, 0x66, 0x76, 0x37, 0x07, 0x00, 0x0f, 0x3f, 0x7f,
0x66, 0x66, 0x66, 0x66, 0x77, 0x27, 0x07, 0x00, 0x1f, 0x3f, 0x76, 0x66, 0x66, 0x66, 0x76, 0x37,
0x07, 0x00, 0x0f, 0x3f, 0x71, 0x60, 0x60, 0x60, 0x60, 0x31, 0x7f, 0x7f, 0x00, 0x00, 0x00, 0x00,
0x11, 0x37, 0x67, 0x66, 0x66, 0x6c, 0x7d, 0x38, 0x00, 0x00, 0x3f, 0x7f, 0x3f, 0x00, 0x00, 0x1f,
0x3f, 0x70, 0x60, 0x60, 0x70, 0x7f, 0x7f, 0x00, 0x0f, 0x3f, 0x71, 0x60, 0x60, 0x60, 0x60, 0x31,
0x7f, 0x7f, 0x00, 0x7f, 0x7f, 0x00, 0x06, 0x1f, 0x3b, 0x60, 0x60, 0x60, 0x60, 0x71, 0x3f, 0x1f,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01,
0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x48, 0x48, 0x48, 0xb0, 0x00, 0xc0, 0x20,
0x20, 0x20, 0xc0, 0x00, 0xc0, 0x20, 0x20, 0x20, 0xc0, 0x00, 0x40, 0xa0, 0xa0, 0xa0, 0x20, 0x00,
0x00, 0x20, 0xf0, 0x20, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0xf8, 0x08,
0x08, 0x00, 0xc0, 0x20, 0x20, 0x20, 0xf8, 0x00, 0xc0, 0xa0, 0xa0, 0xa0, 0xc0, 0x00, 0x20, 0xa0,
0xa0, 0xa0, 0xc0, 0x00, 0x40, 0xa0, 0xa0, 0xa0, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x48, 0x48, 0x48, 0x08, 0x00, 0x20, 0x40, 0x80, 0x40,
0x20, 0x00, 0x00, 0x20, 0xf0, 0x20, 0x20, 0x00, 0xc0, 0xa0, 0xa0, 0xa0, 0xc0, 0x00, 0xe0, 0x00,
0x20, 0x20, 0xc0, 0x00, 0xc0, 0x20, 0x20, 0x20, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x02, 0x02, 0x02, 0x01, 0x00, 0x01, 0x02,
0x02, 0x02, 0x01, 0x00, 0x01, 0x02, 0x02, 0x02, 0x01, 0x00, 0x02, 0x02, 0x02, 0x02, 0x01, 0x00,
0x00, 0x00, 0x01, 0x02, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x02, 0x03, 0x02,
0x02, 0x00, 0x01, 0x02, 0x02, 0x02, 0x03, 0x00, 0x01, 0x02, 0x02, 0x02, 0x00, 0x00, 0x01, 0x02,
0x02, 0x02, 0x01, 0x02, 0x02, 0x02, 0x02, 0x02, 0x01, 0x00, 0x00, 0x08, 0x06, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x02, 0x02, 0x82, 0x02, 0x00, 0x02, 0x01, 0x01, 0x01,
0x02, 0x00, 0x00, 0x00, 0x01, 0x02, 0x02, 0x00, 0x01, 0x02, 0x02, 0x02, 0x00, 0x00, 0x03, 0x00,
0x00, 0x00, 0x03, 0x00, 0x01, 0x02, 0x02, 0x02, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x82, 0x8c, 0x60, 0x1c, 0x02, 0x00, 0x1c, 0x22, 0x22, 0x22, 0x1c, 0x00, 0x1e,
0x20, 0x20, 0x00, 0x3e, 0x00, 0x00, 0x3e, 0x04, 0x02, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x3e, 0x04, 0x02, 0x02, 0x00, 0x1c, 0x2a, 0x2a, 0x2a, 0x0c, 0x00, 0x12, 0x2a, 0x2a,
0x2a, 0x1c, 0x20, 0x1c, 0x22, 0x22, 0x22, 0x14, 0x00, 0x3f, 0x00, 0x02, 0x02, 0x3c, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/*------------------------------------------------------------------------------
    SeeedOLED_init
------------------------------------------------------------------------------*/
void SeeedOLED_init(void)
{
    SeeedOLED_sendCommand(SeeedOLED_Display_Off_Cmd);     // display off
    AbsDelay(5);
    SeeedOLED_sendCommand(SeeedOLED_Display_On_Cmd);      // display on
    AbsDelay(5);
    SeeedOLED_sendCommand(SeeedOLED_Normal_Display_Cmd);  // Set Normal Display (default)
    SeeedOLED_deactivateScroll();
    SeeedOLED_clearDisplay();
}

/*------------------------------------------------------------------------------
    SeeedOLED_sendCommand
------------------------------------------------------------------------------*/
void SeeedOLED_sendCommand(byte command)
{
#if AVR32
    somedata[0]=SeeedOLED_Command_Mode;
    somedata[1]=command;
    I2C_Write(SeeedOLED_Address_AVR32, 0, 0, somedata, 2);
#else
    I2C_Start();
    I2C_Write(SeeedOLED_Address_MEGA);
    I2C_Write(SeeedOLED_Command_Mode);
    I2C_Write(command);
    I2C_Stop();
#endif
}

/*------------------------------------------------------------------------------
    SeeedOLED_setBrightness
------------------------------------------------------------------------------*/
void SeeedOLED_setBrightness(byte Brightness)
{
   SeeedOLED_sendCommand(SeeedOLED_Set_Brightness_Cmd);
   SeeedOLED_sendCommand(Brightness);
}

/*------------------------------------------------------------------------------
    SeeedOLED_setHorizontalMode
------------------------------------------------------------------------------*/
void SeeedOLED_setHorizontalMode(void)
{
    addressingMode = HORIZONTAL_MODE;
    SeeedOLED_sendCommand(0x20);          // set addressing mode
    SeeedOLED_sendCommand(0x00);          // set horizontal addressing mode
}

/*------------------------------------------------------------------------------
    SeeedOLED_setPageMode
------------------------------------------------------------------------------*/
void SeeedOLED_setPageMode(void)
{
    addressingMode = PAGE_MODE;
    SeeedOLED_sendCommand(0x20);          // set addressing mode
    SeeedOLED_sendCommand(0x02);          // set page addressing mode
}

/*------------------------------------------------------------------------------
    SeeedOLED_setTextXY
------------------------------------------------------------------------------*/
void SeeedOLED_setTextXY(unsigned char Row, unsigned char Column)
{
    SeeedOLED_sendCommand(0xB0 + Row);                      // set page address
    SeeedOLED_sendCommand(0x00 + (8*Column & 0x0F));        // set column lower address
    SeeedOLED_sendCommand(0x10 + ((8*Column>>4)&0x0F));     // set column higher address
}

/*------------------------------------------------------------------------------
    SeeedOLED_clearDisplay
------------------------------------------------------------------------------*/
void SeeedOLED_clearDisplay(void)
{
  byte i,j;
  i=0;
  j=0;

  SeeedOLED_sendCommand(SeeedOLED_Display_Off_Cmd);   // display off

  for(j=0;j<8;j++)
  {
    SeeedOLED_setTextXY(j,0);
    {
      for(i=0;i<16;i++)  // clear all columns
      {
        SeeedOLED_putChar(' ');
      }
    }
  }
  SeeedOLED_sendCommand(SeeedOLED_Display_On_Cmd);    // display on
  SeeedOLED_setTextXY(0,0);
}

/*------------------------------------------------------------------------------
    SeeedOLED_sendData
------------------------------------------------------------------------------*/
void SeeedOLED_sendData(byte Data)
{
#if AVR32
    somedata[0]=SeeedOLED_Data_Mode;
    somedata[1]=Data;
    I2C_Write(SeeedOLED_Address_AVR32, 0, 0, somedata, 2);
#else
     I2C_Start();
     I2C_Write(SeeedOLED_Address_MEGA);
     I2C_Write(SeeedOLED_Data_Mode);
     I2C_Write(Data);
     I2C_Stop();
#endif
}

/*------------------------------------------------------------------------------
    SeeedOLED_putChar
------------------------------------------------------------------------------*/
void SeeedOLED_putChar(byte C)
{
    if(C < 32 || C > 127) //Ignore non-printable ASCII characters. This can be modified for multilingual font.
    {
    C=' '; //Space
    }
    byte i;
    for(i=0;i<8;i++)
    {
       //read bytes from code memory
       SeeedOLED_sendData(BasicFont[C-32][i]); //font array starts at 0, ASCII starts at 32. Hence the translation
    }
}

/*------------------------------------------------------------------------------
    SeeedOLED_putString
------------------------------------------------------------------------------*/
void SeeedOLED_putString(byte String[])
{
    byte i;
    i=0;
    while(String[i])
    {
        SeeedOLED_putChar(String[i]);
        i++;
    }
}

/*------------------------------------------------------------------------------
    SeeedOLED_putNumber
------------------------------------------------------------------------------*/
byte SeeedOLED_putNumber(long long_num)
{
  char char_buffer[10];
  byte i;
  byte f;
  i=0;
  f=0;

  if(long_num < 0)
  {
    f=1;
    SeeedOLED_putChar('-');
    long_num = -long_num;
  }
  else if (long_num == 0)
  {
    f=1;
    SeeedOLED_putChar('0');
    return f;
  }

  while(long_num > 0)
  {
    char_buffer[i++] = long_num % 10;
    long_num = long_num / 10;
  }

  f=f+i;
  for(; i > 0; i--)
  {
    SeeedOLED_putChar('0'+ char_buffer[i - 1]);
  }
  return f;

}

/*------------------------------------------------------------------------------
    SeeedOLED_putFloat
------------------------------------------------------------------------------*/
byte SeeedOLED_putFloat(float floatNumber, byte decimal)
{
  unsigned int temp;
  float decy;
  float rounding;
  rounding=0.5;
  byte f;
  int i;

  if(floatNumber<0.0)
  {
    SeeedOLED_putString("-");
    floatNumber = -floatNumber;
    f++;
  }
  for (i=0; i<decimal; ++i)
  {
    rounding = rounding / 10.0;
  }
    floatNumber = floatNumber + rounding;

  temp = floatNumber;
  f = f + SeeedOLED_putNumber(temp);
  if(decimal>0)
  {
    SeeedOLED_putChar('.');
    f++;
 }
  decy = floatNumber-temp;  // decimal part,
  for(i=0;i<decimal;i++)    // 4
  {
    decy = decy * 10;       // for the next decimal
    temp = decy;            // get the decimal
    SeeedOLED_putNumber(temp);
    decy = decy-temp;
  }
  f = f + decimal;
  return f;
}

/*------------------------------------------------------------------------------
    SeeedOLED_drawBitmap
------------------------------------------------------------------------------*/
void SeeedOLED_drawBitmap(void)
{
  char localAddressMode;
  int i;

  localAddressMode = addressingMode;
  if(addressingMode != HORIZONTAL_MODE)
  {
      // Bitmap is drawn in horizontal mode
      SeeedOLED_setHorizontalMode();
  }

  for(i=0;i<1024;i++)
  {
      SeeedOLED_sendData(SeeedLogo[i]);
  }

  if(localAddressMode == PAGE_MODE)
  {
     // If pageMode was used earlier, restore it.
     SeeedOLED_setPageMode();
  }
}

/*------------------------------------------------------------------------------
    SeeedOLED_setHorizontalScrollProperties
------------------------------------------------------------------------------*/
void SeeedOLED_setHorizontalScrollProperties(byte direction, byte startPage, byte endPage, byte scrollSpeed)
{
/*
Use the following defines for 'direction' :

 Scroll_Left
 Scroll_Right

Use the following defines for 'scrollSpeed' :

 Scroll_2Frames
 Scroll_3Frames
 Scroll_4Frames
 Scroll_5Frames
 Scroll_25Frames
 Scroll_64Frames
 Scroll_128Frames
 Scroll_256Frames

*/

   if(Scroll_Right == direction)
   {
        // Scroll Right
        SeeedOLED_sendCommand(0x26);
   }
   else
   {
        // Scroll Left
        SeeedOLED_sendCommand(0x27);

   }

   SeeedOLED_sendCommand(0x00);
   SeeedOLED_sendCommand(startPage);
   SeeedOLED_sendCommand(scrollSpeed);
   SeeedOLED_sendCommand(endPage);
   SeeedOLED_sendCommand(0x00);
   SeeedOLED_sendCommand(0xFF);
}

/*------------------------------------------------------------------------------
    SeeedOLED_activateScroll
------------------------------------------------------------------------------*/
void SeeedOLED_activateScroll(void)
{
    SeeedOLED_sendCommand(SeeedOLED_Activate_Scroll_Cmd);
}

/*------------------------------------------------------------------------------
    SeeedOLED_deactivateScroll
------------------------------------------------------------------------------*/
void SeeedOLED_deactivateScroll(void)
{
    SeeedOLED_sendCommand(SeeedOLED_Dectivate_Scroll_Cmd);
}

/*------------------------------------------------------------------------------
    SeeedOLED_setNormalDisplay
------------------------------------------------------------------------------*/
void SeeedOLED_setNormalDisplay(void)
{
    SeeedOLED_sendCommand(SeeedOLED_Normal_Display_Cmd);
}

/*------------------------------------------------------------------------------
    SeeedOLED_setInverseDisplay
------------------------------------------------------------------------------*/
void SeeedOLED_setInverseDisplay(void)
{
    SeeedOLED_sendCommand(SeeedOLED_Inverse_Display_Cmd);
}

/*------------------------------------------------------------------------------
    Main program
------------------------------------------------------------------------------*/
void main(void)
{
    #if AVR32
        // Notihing...
    #else
        // MEGA Serie
        I2C_Init(I2C_400kHz);           // I2C Bit Rate: 100 kHz
    #endif

    SeeedOLED_init();

    // Draw Bitmap Logo and scroll the Logo left and right
    SeeedOLED_clearDisplay();
    SeeedOLED_drawBitmap();
    AbsDelay(2500);
    SeeedOLED_setHorizontalScrollProperties(Scroll_Left,4,7,Scroll_5Frames);  //Set the properties of Horizontal Scrool
    SeeedOLED_activateScroll();         // Activate Scroll
    AbsDelay(8000);
    SeeedOLED_setHorizontalScrollProperties(Scroll_Right,4,7,Scroll_5Frames); //Set Scrolling properties to Scroll Right
    SeeedOLED_activateScroll();         // Activate Scrolling
    AbsDelay(8000);
    SeeedOLED_deactivateScroll();

    // Show numbers
    SeeedOLED_clearDisplay();           // clear the screen and set start position to top left corner
    SeeedOLED_setNormalDisplay();       // Set display to Normal mode
    SeeedOLED_setPageMode();            // Set addressing mode to Page Mode
    SeeedOLED_setTextXY(0,0);           // Set the cursor to 0th Page, 0th Column
    SeeedOLED_putNumber(123);           // Print number
    SeeedOLED_setTextXY(1,0);           // Set the cursor to 1st Page, 0th Column
    SeeedOLED_putNumber(0xFFFF);        // Print number
    SeeedOLED_setTextXY(2,0);           // Set the cursor to 2nd Page, 0th Column
    SeeedOLED_putNumber(0xFFFFFFFF);    // Print number
    SeeedOLED_setTextXY(3,0);           // Set the cursor to 3rd Page, 0th Column
    SeeedOLED_putNumber(-1234L);        // Print number
    SeeedOLED_setTextXY(4,0);           // Set the cursor To 4rd Page, 0th Column#
    SeeedOLED_putFloat(3.1415,4);       // Print number
    SeeedOLED_setTextXY(5,0);           // Set the cursor To 4rd Page, 0th Column
    SeeedOLED_putFloat(-42.545454,2);   // Print number
    AbsDelay(5000);

    // Show Text
    SeeedOLED_clearDisplay();           // clear the screen and set start position to top left corner
    SeeedOLED_setNormalDisplay();       // Set display to normal mode (i.e non-inverse mode)
    SeeedOLED_setPageMode();            // Set addressing mode to Page Mode
    SeeedOLED_setTextXY(0,2);           // Set the cursor to Xth Page, Yth Column
    SeeedOLED_putString("Hello World!");// Print the String
    AbsDelay(5000);

    // Show Misc
    SeeedOLED_clearDisplay();           // clear the screen and set start position to top left corner
    SeeedOLED_setNormalDisplay();       // Set display to Normal mode
    SeeedOLED_setHorizontalMode();      // Set addressing mode to Horizontal Mode
    SeeedOLED_putString("!\#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~");  //Print String (ASCII 32 - 126 )

}

/*******************************************************************************
 * Info
 *******************************************************************************
 * Changelog:
 * -
 *
 *******************************************************************************
 * Bugs, feedback, questions and modifications can be posted on the
 * C-Control Forum on http://www.c-control.de
 * Of course you can also write us an e-mail to: webmaster@c-control.de
 * We publish updates from time to time on www.c-control.de!
/******************************************************************************/

// EOF

